#!/usr/bin/env python3
"""
TuskLang Documentation Asset Generator
Generates all SVG assets for the TuskLang Python SDK documentation
"""

import os
import json
from pathlib import Path
from typing import Dict, List, Tuple

class TuskLangAssetGenerator:
    """Generate all SVG assets for TuskLang documentation"""
    
    def __init__(self, base_path: str = "."):
        self.base_path = Path(base_path)
        self.assets_path = self.base_path / "assets"
        self.icons_path = self.assets_path / "icons"
        
        # TuskLang color palette
        self.colors = {
            'primary': '#FF6B6B',      # Coral Red
            'secondary': '#4ECDC4',    # Turquoise
            'accent': '#667eea',       # Purple
            'success': '#2ecc71',      # Green
            'warning': '#f39c12',      # Orange
            'danger': '#e74c3c',       # Red
            'info': '#3498db',         # Blue
            'dark': '#2c3e50',         # Dark Blue
            'gray': '#7f8c8d',         # Gray
            'light': '#ecf0f1',        # Light Gray
            'purple': '#9b59b6',       # Purple
        }
    
    def setup_directories(self):
        """Create directory structure"""
        directories = [
            self.assets_path,
            self.icons_path,
            self.assets_path / "diagrams",
            self.assets_path / "charts",
        ]
        
        for directory in directories:
            directory.mkdir(parents=True, exist_ok=True)
        
        print("✅ Directory structure created")
    
    def create_logo(self):
        """Create main TuskLang logo"""
        logo_svg = f'''<svg width="600" height="150" viewBox="0 0 600 150" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="logoGradient" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['primary']};stop-opacity:1" />
      <stop offset="50%" style="stop-color:{self.colors['secondary']};stop-opacity:1" />
      <stop offset="100%" style="stop-color:{self.colors['accent']};stop-opacity:1" />
    </linearGradient>
    <filter id="logoShadow">
      <feDropShadow dx="0" dy="4" stdDeviation="4" flood-opacity="0.2"/>
    </filter>
    <pattern id="gridPattern" width="4" height="4" patternUnits="userSpaceOnUse">
      <circle cx="2" cy="2" r="0.5" fill="#fff" opacity="0.3"/>
    </pattern>
  </defs>
  
  <!-- Logo Symbol -->
  <g transform="translate(50, 30)">
    <!-- Outer hexagon -->
    <path d="M50 10 L80 25 L80 55 L50 70 L20 55 L20 25 Z" 
          fill="url(#logoGradient)" filter="url(#logoShadow)"/>
    
    <!-- Inner design -->
    <path d="M50 20 L70 30 L70 50 L50 60 L30 50 L30 30 Z" 
          fill="url(#gridPattern)" opacity="0.8"/>
    
    <!-- Center T shape -->
    <path d="M40 30 L60 30 L60 35 L52.5 35 L52.5 50 L47.5 50 L47.5 35 L40 35 Z" 
          fill="#fff"/>
    
    <!-- Orbit dots -->
    <circle cx="50" cy="15" r="3" fill="#fff" opacity="0.8">
      <animate attributeName="opacity" values="0.8;1;0.8" dur="2s" repeatCount="indefinite"/>
    </circle>
    <circle cx="75" cy="40" r="3" fill="#fff" opacity="0.8">
      <animate attributeName="opacity" values="0.8;1;0.8" dur="2s" begin="0.5s" repeatCount="indefinite"/>
    </circle>
    <circle cx="25" cy="40" r="3" fill="#fff" opacity="0.8">
      <animate attributeName="opacity" values="0.8;1;0.8" dur="2s" begin="1s" repeatCount="indefinite"/>
    </circle>
  </g>
  
  <!-- Text -->
  <text x="150" y="70" font-family="Arial, sans-serif" font-size="56" font-weight="bold" fill="{self.colors['dark']}">
    TuskLang
  </text>
  <text x="150" y="100" font-family="Arial, sans-serif" font-size="18" fill="{self.colors['gray']}">
    Python SDK • 200+ Functions • Enterprise Scale
  </text>
</svg>'''
        
        self._save_svg('tusklang-logo.svg', logo_svg)
    
    def create_hero_banner(self):
        """Create hero section banner"""
        hero_svg = f'''<svg width="1200" height="400" viewBox="0 0 1200 400" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="heroBg" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['accent']};stop-opacity:1" />
      <stop offset="50%" style="stop-color:#764ba2;stop-opacity:1" />
      <stop offset="100%" style="stop-color:#f093fb;stop-opacity:1" />
    </linearGradient>
    <filter id="glow">
      <feGaussianBlur stdDeviation="4" result="coloredBlur"/>
      <feMerge>
        <feMergeNode in="coloredBlur"/>
        <feMergeNode in="SourceGraphic"/>
      </feMerge>
    </filter>
    <pattern id="hexGrid" width="60" height="52" patternUnits="userSpaceOnUse">
      <path d="M30 0 L50 10 L50 30 L30 40 L10 30 L10 10 Z" 
            fill="none" stroke="#fff" stroke-width="0.5" opacity="0.1"/>
    </pattern>
  </defs>
  
  <!-- Background -->
  <rect width="1200" height="400" fill="url(#heroBg)"/>
  <rect width="1200" height="400" fill="url(#hexGrid)"/>
  
  <!-- Floating elements -->
  <g opacity="0.3">
    <circle cx="100" cy="100" r="40" fill="#fff">
      <animate attributeName="cy" values="100;120;100" dur="4s" repeatCount="indefinite"/>
    </circle>
    <circle cx="1100" cy="300" r="60" fill="#fff">
      <animate attributeName="cy" values="300;280;300" dur="5s" repeatCount="indefinite"/>
    </circle>
    <circle cx="600" cy="50" r="30" fill="#fff">
      <animate attributeName="cy" values="50;70;50" dur="3s" repeatCount="indefinite"/>
    </circle>
  </g>
  
  <!-- Main content -->
  <g transform="translate(600, 200)" text-anchor="middle">
    <!-- 200+ Functions -->
    <g transform="translate(-300, -50)">
      <text font-family="Arial" font-size="72" font-weight="bold" fill="#fff" filter="url(#glow)">200+</text>
      <text y="30" font-family="Arial" font-size="20" fill="#fff" opacity="0.9">Functions</text>
    </g>
    
    <!-- 85 Operators -->
    <g transform="translate(0, -50)">
      <text font-family="Arial" font-size="72" font-weight="bold" fill="#fff" filter="url(#glow)">85</text>
      <text y="30" font-family="Arial" font-size="20" fill="#fff" opacity="0.9">Operators</text>
    </g>
    
    <!-- 24x Faster -->
    <g transform="translate(300, -50)">
      <text font-family="Arial" font-size="72" font-weight="bold" fill="#fff" filter="url(#glow)">24x</text>
      <text y="30" font-family="Arial" font-size="20" fill="#fff" opacity="0.9">Faster</text>
    </g>
  </g>
  
  <!-- Tagline -->
  <text x="600" y="320" font-family="Arial" font-size="24" fill="#fff" text-anchor="middle" opacity="0.9">
    Enterprise Configuration Management at Scale
  </text>
  
  <!-- Code snippet preview -->
  <g transform="translate(350, 350)" opacity="0.6">
    <rect x="0" y="0" width="500" height="40" rx="20" fill="#fff" opacity="0.2"/>
    <text x="250" y="25" font-family="monospace" font-size="16" fill="#fff" text-anchor="middle">
      pip install tusktsk[full]
    </text>
  </g>
</svg>'''
        
        self._save_svg('hero-banner.svg', hero_svg)
    
    def create_function_dashboard(self):
        """Create function categories dashboard"""
        dashboard_svg = f'''<svg width="900" height="500" viewBox="0 0 900 500" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="dashGrad1" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['primary']};stop-opacity:1" />
      <stop offset="100%" style="stop-color:#ee5a6f;stop-opacity:1" />
    </linearGradient>
    <linearGradient id="dashGrad2" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['secondary']};stop-opacity:1" />
      <stop offset="100%" style="stop-color:#44a2bb;stop-opacity:1" />
    </linearGradient>
    <linearGradient id="dashGrad3" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['warning']};stop-opacity:1" />
      <stop offset="100%" style="stop-color:#e67e22;stop-opacity:1" />
    </linearGradient>
    <filter id="cardShadow">
      <feDropShadow dx="0" dy="4" stdDeviation="8" flood-opacity="0.15"/>
    </filter>
  </defs>
  
  <!-- Background -->
  <rect width="900" height="500" fill="#f8f9fa"/>
  
  <!-- Grid pattern -->
  <defs>
    <pattern id="dashGrid" width="20" height="20" patternUnits="userSpaceOnUse">
      <rect width="20" height="20" fill="none" stroke="#e0e0e0" stroke-width="0.5"/>
    </pattern>
  </defs>
  <rect width="900" height="500" fill="url(#dashGrid)" opacity="0.5"/>
  
  <!-- Title -->
  <text x="450" y="40" font-family="Arial" font-size="28" font-weight="bold" fill="{self.colors['dark']}" text-anchor="middle">
    Function Categories Overview
  </text>
  
  <!-- Category cards -->
  <!-- Row 1 -->
  <g transform="translate(50, 80)">
    <!-- Core Operators -->
    <rect width="250" height="150" rx="10" fill="url(#dashGrad1)" filter="url(#cardShadow)"/>
    <text x="20" y="35" font-family="Arial" font-size="20" font-weight="bold" fill="#fff">Core Operators</text>
    <text x="20" y="80" font-family="Arial" font-size="48" font-weight="bold" fill="#fff">85</text>
    <text x="20" y="110" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
      @env @cache @date @file
    </text>
    <text x="20" y="130" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
      @json @query @variable
    </text>
    
    <!-- Database -->
    <g transform="translate(275, 0)">
      <rect width="250" height="150" rx="10" fill="url(#dashGrad2)" filter="url(#cardShadow)"/>
      <text x="20" y="35" font-family="Arial" font-size="20" font-weight="bold" fill="#fff">Database</text>
      <text x="20" y="80" font-family="Arial" font-size="48" font-weight="bold" fill="#fff">50+</text>
      <text x="20" y="110" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
        SQLite PostgreSQL MySQL
      </text>
      <text x="20" y="130" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
        MongoDB Redis Multi-DB
      </text>
    </g>
    
    <!-- Communication -->
    <g transform="translate(550, 0)">
      <rect width="250" height="150" rx="10" fill="url(#dashGrad3)" filter="url(#cardShadow)"/>
      <text x="20" y="35" font-family="Arial" font-size="20" font-weight="bold" fill="#fff">Communication</text>
      <text x="20" y="80" font-family="Arial" font-size="48" font-weight="bold" fill="#fff">30+</text>
      <text x="20" y="110" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
        WebSocket GraphQL gRPC
      </text>
      <text x="20" y="130" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
        Slack Teams Email SMS
      </text>
    </g>
  </g>
  
  <!-- Row 2 -->
  <g transform="translate(50, 250)">
    <!-- Security -->
    <rect width="250" height="150" rx="10" fill="{self.colors['success']}" filter="url(#cardShadow)"/>
    <text x="20" y="35" font-family="Arial" font-size="20" font-weight="bold" fill="#fff">Security</text>
    <text x="20" y="80" font-family="Arial" font-size="48" font-weight="bold" fill="#fff">25+</text>
    <text x="20" y="110" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
      OAuth2 JWT Encryption
    </text>
    <text x="20" y="130" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
      RBAC Audit Compliance
    </text>
    
    <!-- Cloud -->
    <g transform="translate(275, 0)">
      <rect width="250" height="150" rx="10" fill="{self.colors['info']}" filter="url(#cardShadow)"/>
      <text x="20" y="35" font-family="Arial" font-size="20" font-weight="bold" fill="#fff">Cloud</text>
      <text x="20" y="80" font-family="Arial" font-size="48" font-weight="bold" fill="#fff">20+</text>
      <text x="20" y="110" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
        AWS Azure GCP
      </text>
      <text x="20" y="130" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
        Kubernetes Docker Terraform
      </text>
    </g>
    
    <!-- AI/ML -->
    <g transform="translate(550, 0)">
      <rect width="250" height="150" rx="10" fill="{self.colors['purple']}" filter="url(#cardShadow)"/>
      <text x="20" y="35" font-family="Arial" font-size="20" font-weight="bold" fill="#fff">AI/ML</text>
      <text x="20" y="80" font-family="Arial" font-size="48" font-weight="bold" fill="#fff">15+</text>
      <text x="20" y="110" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
        AutoML Neural Networks
      </text>
      <text x="20" y="130" font-family="Arial" font-size="14" fill="#fff" opacity="0.9">
        NLP Computer Vision
      </text>
    </g>
  </g>
  
  <!-- Total counter -->
  <g transform="translate(450, 440)">
    <circle r="35" fill="{self.colors['danger']}" filter="url(#cardShadow)"/>
    <text font-family="Arial" font-size="20" font-weight="bold" fill="#fff" text-anchor="middle">200+</text>
    <text y="20" font-family="Arial" font-size="10" fill="#fff" text-anchor="middle">Total</text>
  </g>
</svg>'''
        
        self._save_svg('function-dashboard.svg', dashboard_svg)
    
    def create_performance_chart(self):
        """Create performance comparison chart"""
        perf_svg = f'''<svg width="800" height="500" viewBox="0 0 800 500" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="perfGrad1" x1="0%" y1="0%" x2="0%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['danger']};stop-opacity:1" />
      <stop offset="100%" style="stop-color:#c0392b;stop-opacity:1" />
    </linearGradient>
    <linearGradient id="perfGrad2" x1="0%" y1="0%" x2="0%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['success']};stop-opacity:1" />
      <stop offset="100%" style="stop-color:#27ae60;stop-opacity:1" />
    </linearGradient>
    <filter id="barShadow">
      <feDropShadow dx="2" dy="2" stdDeviation="2" flood-opacity="0.2"/>
    </filter>
  </defs>
  
  <!-- Background -->
  <rect width="800" height="500" fill="#f8f9fa"/>
  
  <!-- Title -->
  <text x="400" y="40" font-family="Arial" font-size="24" font-weight="bold" 
        text-anchor="middle" fill="{self.colors['dark']}">Performance Benchmarks</text>
  
  <!-- Chart area -->
  <g transform="translate(100, 80)">
    <!-- Y-axis -->
    <line x1="0" y1="0" x2="0" y2="320" stroke="{self.colors['gray']}" stroke-width="2"/>
    <!-- X-axis -->
    <line x1="0" y1="320" x2="600" y2="320" stroke="{self.colors['gray']}" stroke-width="2"/>
    
    <!-- Y-axis labels -->
    <text x="-10" y="5" font-family="Arial" font-size="12" text-anchor="end" fill="{self.colors['gray']}">30x</text>
    <text x="-10" y="65" font-family="Arial" font-size="12" text-anchor="end" fill="{self.colors['gray']}">25x</text>
    <text x="-10" y="125" font-family="Arial" font-size="12" text-anchor="end" fill="{self.colors['gray']}">20x</text>
    <text x="-10" y="185" font-family="Arial" font-size="12" text-anchor="end" fill="{self.colors['gray']}">15x</text>
    <text x="-10" y="245" font-family="Arial" font-size="12" text-anchor="end" fill="{self.colors['gray']}">10x</text>
    <text x="-10" y="305" font-family="Arial" font-size="12" text-anchor="end" fill="{self.colors['gray']}">5x</text>
    <text x="-10" y="325" font-family="Arial" font-size="12" text-anchor="end" fill="{self.colors['gray']}">1x</text>
    
    <!-- Bars -->
    <!-- Config Parsing: 24x -->
    <g transform="translate(50, 0)">
      <rect x="0" y="64" width="80" height="256" fill="url(#perfGrad2)" filter="url(#barShadow)"/>
      <text x="40" y="50" font-family="Arial" font-size="20" font-weight="bold" 
            text-anchor="middle" fill="{self.colors['success']}">24x</text>
      <text x="40" y="340" font-family="Arial" font-size="12" text-anchor="middle" fill="{self.colors['dark']}">Config</text>
      <text x="40" y="355" font-family="Arial" font-size="12" text-anchor="middle" fill="{self.colors['dark']}">Parsing</text>
    </g>
    
    <!-- Database Query: 7x -->
    <g transform="translate(180, 0)">
      <rect x="0" y="240" width="80" height="80" fill="url(#perfGrad2)" filter="url(#barShadow)"/>
      <text x="40" y="226" font-family="Arial" font-size="20" font-weight="bold" 
            text-anchor="middle" fill="{self.colors['success']}">7x</text>
      <text x="40" y="340" font-family="Arial" font-size="12" text-anchor="middle" fill="{self.colors['dark']}">Database</text>
      <text x="40" y="355" font-family="Arial" font-size="12" text-anchor="middle" fill="{self.colors['dark']}">Query</text>
    </g>
    
    <!-- Binary Loading: 17x -->
    <g transform="translate(310, 0)">
      <rect x="0" y="128" width="80" height="192" fill="url(#perfGrad2)" filter="url(#barShadow)"/>
      <text x="40" y="114" font-family="Arial" font-size="20" font-weight="bold" 
            text-anchor="middle" fill="{self.colors['success']}">17x</text>
      <text x="40" y="340" font-family="Arial" font-size="12" text-anchor="middle" fill="{self.colors['dark']}">Binary</text>
      <text x="40" y="355" font-family="Arial" font-size="12" text-anchor="middle" fill="{self.colors['dark']}">Loading</text>
    </g>
    
    <!-- Template Render: 30x -->
    <g transform="translate(440, 0)">
      <rect x="0" y="0" width="80" height="320" fill="url(#perfGrad2)" filter="url(#barShadow)"/>
      <text x="40" y="-14" font-family="Arial" font-size="20" font-weight="bold" 
            text-anchor="middle" fill="{self.colors['success']}">30x</text>
      <text x="40" y="340" font-family="Arial" font-size="12" text-anchor="middle" fill="{self.colors['dark']}">Template</text>
      <text x="40" y="355" font-family="Arial" font-size="12" text-anchor="middle" fill="{self.colors['dark']}">Render</text>
    </g>
  </g>
  
  <!-- Legend -->
  <g transform="translate(300, 450)">
    <rect x="0" y="0" width="15" height="15" fill="url(#perfGrad1)"/>
    <text x="20" y="12" font-family="Arial" font-size="12" fill="{self.colors['gray']}">Standard Python</text>
    <rect x="150" y="0" width="15" height="15" fill="url(#perfGrad2)"/>
    <text x="170" y="12" font-family="Arial" font-size="12" fill="{self.colors['gray']}">TuskLang SDK</text>
  </g>
</svg>'''
        
        self._save_svg('performance-comparison.svg', perf_svg)
    
    def create_icons(self):
        """Create all icon SVGs"""
        icons = {
            'functions': {
                'color': self.colors['primary'],
                'content': '<text x="40" y="35" font-family="Arial" font-size="20" font-weight="bold" fill="#fff" text-anchor="middle">f(x)</text><text x="40" y="55" font-family="Arial" font-size="14" fill="#fff" text-anchor="middle">200+</text>'
            },
            'operators': {
                'color': self.colors['secondary'],
                'content': '<text x="40" y="48" font-family="Arial" font-size="32" font-weight="bold" fill="#fff" text-anchor="middle">@</text>'
            },
            'enterprise': {
                'color': self.colors['accent'],
                'content': '''<rect x="15" y="20" width="50" height="40" rx="5" fill="#fff" opacity="0.3"/>
                <rect x="25" y="30" width="10" height="10" fill="#fff"/>
                <rect x="45" y="30" width="10" height="10" fill="#fff"/>
                <rect x="25" y="45" width="10" height="10" fill="#fff"/>
                <rect x="45" y="45" width="10" height="10" fill="#fff"/>'''
            },
            'performance': {
                'color': self.colors['warning'],
                'content': '''<path d="M25 50 L35 30 L45 45 L55 25" stroke="#fff" stroke-width="4" fill="none" stroke-linecap="round" stroke-linejoin="round"/>
                <circle cx="55" cy="25" r="3" fill="#fff"/>'''
            },
            'ai': {
                'color': self.colors['purple'],
                'content': '''<circle cx="40" cy="40" r="12" fill="none" stroke="#fff" stroke-width="3"/>
                <circle cx="40" cy="25" r="5" fill="#fff"/>
                <circle cx="55" cy="40" r="5" fill="#fff"/>
                <circle cx="40" cy="55" r="5" fill="#fff"/>
                <circle cx="25" cy="40" r="5" fill="#fff"/>'''
            },
            'database': {
                'color': self.colors['info'],
                'content': '''<ellipse cx="40" cy="25" rx="25" ry="10" fill="#fff" opacity="0.3"/>
                <path d="M15 25 L15 55 Q40 70 65 55 L65 25" fill="#fff" opacity="0.3"/>
                <ellipse cx="40" cy="55" rx="25" ry="10" fill="#fff" opacity="0.5"/>'''
            },
            'security': {
                'color': self.colors['success'],
                'content': '''<path d="M40 15 L60 25 L60 45 Q60 65 40 70 Q20 65 20 45 L20 25 Z" fill="#fff" opacity="0.3"/>
                <path d="M30 40 L37 47 L50 32" stroke="#fff" stroke-width="4" fill="none" stroke-linecap="round" stroke-linejoin="round"/>'''
            },
            'cloud': {
                'color': self.colors['info'],
                'content': '''<path d="M50 35 Q50 25 40 25 Q30 25 30 35 Q20 35 20 45 Q20 55 30 55 L50 55 Q60 55 60 45 Q60 35 50 35 Z" fill="#fff" opacity="0.3"/>
                <circle cx="30" cy="40" r="3" fill="#fff"/>
                <circle cx="40" cy="40" r="3" fill="#fff"/>
                <circle cx="50" cy="40" r="3" fill="#fff"/>'''
            }
        }
        
        for name, icon_data in icons.items():
            icon_svg = f'''<svg width="80" height="80" viewBox="0 0 80 80" xmlns="http://www.w3.org/2000/svg">
  <circle cx="40" cy="40" r="35" fill="{icon_data['color']}"/>
  {icon_data['content']}
</svg>'''
            self._save_svg(f'icons/{name}.svg', icon_svg)
        
        # Create additional small icons
        self._create_small_icons()
    
    def _create_small_icons(self):
        """Create small utility icons"""
        small_icons = {
            'check-circle': f'<circle cx="12" cy="12" r="10" stroke="{self.colors["success"]}" stroke-width="2" fill="none"/><path d="M8 12l2 2 4-4" stroke="{self.colors["success"]}" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round"/>',
            'arrow-up': f'<path d="M12 19V5m0 0l-7 7m7-7l7 7" stroke="{self.colors["gray"]}" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>',
            'github': '<path d="M12 2C6.48 2 2 6.48 2 12c0 4.42 2.87 8.17 6.84 9.5.5.08.66-.23.66-.5v-1.69c-2.77.6-3.36-1.34-3.36-1.34-.46-1.16-1.11-1.47-1.11-1.47-.91-.62.07-.6.07-.6 1 .07 1.53 1.03 1.53 1.03.87 1.52 2.34 1.07 2.91.83.09-.65.35-1.09.63-1.34-2.22-.25-4.55-1.11-4.55-4.92 0-1.11.38-2 1.03-2.71-.1-.25-.45-1.29.1-2.64 0 0 .84-.27 2.75 1.02.79-.22 1.65-.33 2.5-.33.85 0 1.71.11 2.5.33 1.91-1.29 2.75-1.02 2.75-1.02.55 1.35.2 2.39.1 2.64.65.71 1.03 1.6 1.03 2.71 0 3.82-2.34 4.66-4.57 4.91.36.31.69.92.69 1.85V21c0 .27.16.59.67.5C19.14 20.16 22 16.42 22 12A10 10 0 0012 2z" fill="currentColor"/>',
            'docs': f'<rect x="4" y="3" width="16" height="18" rx="2" stroke="{self.colors["info"]}" stroke-width="2" fill="none"/><path d="M8 7h8M8 11h8M8 15h5" stroke="{self.colors["info"]}" stroke-width="2" stroke-linecap="round"/>',
            'discord': '<path d="M20.222 0c1.406 0 2.54 1.137 2.607 2.475V24l-2.677-2.273-1.47-1.338-1.604-1.398.67 2.205H3.71c-1.402 0-2.54-1.065-2.54-2.476V2.48C1.17 1.142 2.31.003 3.715.003h16.5L20.222 0zm-6.118 5.683h-.03l-.202.2c2.073.6 3.076 1.537 3.076 1.537-1.336-.668-2.54-1.002-3.744-1.137-.87-.135-1.74-.064-2.475 0h-.2c-.47 0-1.47.2-2.81.735-.467.203-.735.336-.735.336s1.002-1.002 3.21-1.537l-.135-.135s-1.672-.064-3.477 1.27c0 0-1.805 3.144-1.805 7.02 0 0 1.002 1.74 3.744 1.806 0 0 .4-.533.805-1.002-1.54-.468-2.14-1.404-2.14-1.404s.134.066.335.2h.06c.03 0 .044.015.06.03v.006c.016.016.03.03.06.03.33.136.66.27.93.4a8.18 8.18 0 001.8.536 9.04 9.04 0 003.01 0 8.42 8.42 0 001.74-.536c.414-.202.87-.47 1.34-.806 0 0-.67 1.002-2.205 1.47.336.47.805 1.002.805 1.002 2.74-.064 3.81-1.806 3.876-1.74 0-3.876-1.806-7.02-1.806-7.02-1.74-1.268-3.41-1.268-3.41-1.268l-.134.134zm-.27 4.424c.67 0 1.203.602 1.203 1.34 0 .736-.533 1.336-1.203 1.336-.67 0-1.203-.6-1.203-1.337s.533-1.34 1.203-1.34zm-4.305 0c.67 0 1.203.602 1.203 1.34 0 .736-.533 1.336-1.203 1.336-.67 0-1.203-.6-1.203-1.337s.533-1.34 1.203-1.34z" fill="currentColor"/>',
        }
        
        for name, path_data in small_icons.items():
            icon_svg = f'''<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
  {path_data}
</svg>'''
            self._save_svg(f'icons/{name}.svg', icon_svg)
    
    def create_quickstart_flow(self):
        """Create quickstart flow diagram"""
        flow_svg = f'''<svg width="900" height="400" viewBox="0 0 900 400" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <filter id="flowShadow">
      <feDropShadow dx="2" dy="2" stdDeviation="3" flood-opacity="0.2"/>
    </filter>
    <marker id="arrowhead" markerWidth="10" markerHeight="10" refX="9" refY="3" orient="auto">
      <polygon points="0 0, 10 3, 0 6" fill="{self.colors['gray']}"/>
    </marker>
  </defs>
  
  <!-- Background -->
  <rect width="900" height="400" fill="#f8f9fa"/>
  
  <!-- Title -->
  <text x="450" y="40" font-family="Arial" font-size="28" font-weight="bold" 
        text-anchor="middle" fill="{self.colors['dark']}">Get Started in 3 Steps</text>
  
  <!-- Step 1: Install -->
  <g transform="translate(100, 100)">
    <rect width="200" height="200" rx="10" fill="{self.colors['info']}" filter="url(#flowShadow)"/>
    <text x="100" y="40" font-family="Arial" font-size="48" font-weight="bold" 
          fill="#fff" text-anchor="middle">1</text>
    <text x="100" y="80" font-family="Arial" font-size="20" fill="#fff" 
          text-anchor="middle">Install</text>
    
    <!-- Code snippet -->
    <rect x="10" y="100" width="180" height="80" rx="5" fill="{self.colors['dark']}"/>
    <text x="100" y="130" font-family="monospace" font-size="12" fill="{self.colors['success']}" 
          text-anchor="middle">$ pip install</text>
    <text x="100" y="150" font-family="monospace" font-size="12" fill="#fff" 
          text-anchor="middle">tusktsk[full]</text>
    <text x="100" y="170" font-family="monospace" font-size="12" fill="#95a5a6" 
          text-anchor="middle"># Install everything</text>
  </g>
  
  <!-- Arrow 1->2 -->
  <path d="M 310 200 L 340 200" stroke="{self.colors['gray']}" stroke-width="3" 
        marker-end="url(#arrowhead)" stroke-dasharray="5,5">
    <animate attributeName="stroke-dashoffset" from="10" to="0" dur="1s" repeatCount="indefinite"/>
  </path>
  
  <!-- Step 2: Import -->
  <g transform="translate(350, 100)">
    <rect width="200" height="200" rx="10" fill="{self.colors['success']}" filter="url(#flowShadow)"/>
    <text x="100" y="40" font-family="Arial" font-size="48" font-weight="bold" 
          fill="#fff" text-anchor="middle">2</text>
    <text x="100" y="80" font-family="Arial" font-size="20" fill="#fff" 
          text-anchor="middle">Import</text>
    
    <!-- Code snippet -->
    <rect x="10" y="100" width="180" height="80" rx="5" fill="#27ae60"/>
    <text x="100" y="130" font-family="monospace" font-size="12" fill="#fff" 
          text-anchor="middle">from tusktsk</text>
    <text x="100" y="150" font-family="monospace" font-size="12" fill="#fff" 
          text-anchor="middle">import TSK</text>
    <text x="100" y="170" font-family="monospace" font-size="12" fill="#a9dfbf" 
          text-anchor="middle"># Ready to use!</text>
  </g>
  
  <!-- Arrow 2->3 -->
  <path d="M 560 200 L 590 200" stroke="{self.colors['gray']}" stroke-width="3" 
        marker-end="url(#arrowhead)" stroke-dasharray="5,5">
    <animate attributeName="stroke-dashoffset" from="10" to="0" dur="1s" repeatCount="indefinite"/>
  </path>
  
  <!-- Step 3: Use -->
  <g transform="translate(600, 100)">
    <rect width="200" height="200" rx="10" fill="{self.colors['danger']}" filter="url(#flowShadow)"/>
    <text x="100" y="40" font-family="Arial" font-size="48" font-weight="bold" 
          fill="#fff" text-anchor="middle">3</text>
    <text x="100" y="80" font-family="Arial" font-size="20" fill="#fff" 
          text-anchor="middle">Use</text>
    
    <!-- Code snippet -->
    <rect x="10" y="100" width="180" height="80" rx="5" fill="#c0392b"/>
    <text x="100" y="125" font-family="monospace" font-size="11" fill="#fff" 
          text-anchor="middle">tsk = TSK()</text>
    <text x="100" y="145" font-family="monospace" font-size="11" fill="#fff" 
          text-anchor="middle">data = tsk.evaluate(</text>
    <text x="100" y="165" font-family="monospace" font-size="11" fill="#f1c40f" 
          text-anchor="middle">"@cache('key')")</text>
  </g>
  
  <!-- Success message -->
  <g transform="translate(450, 340)">
    <rect x="-100" y="-20" width="200" height="40" rx="20" fill="{self.colors['success']}" filter="url(#flowShadow)"/>
    <text font-family="Arial" font-size="16" font-weight="bold" fill="#fff" text-anchor="middle">
      🚀 You're ready!
    </text>
  </g>
</svg>'''
        
        self._save_svg('quickstart-flow.svg', flow_svg)
    
    def create_footer_wave(self):
        """Create footer wave SVG"""
        wave_svg = f'''<svg width="1200" height="150" viewBox="0 0 1200 150" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="waveGradient" x1="0%" y1="0%" x2="0%" y2="100%">
      <stop offset="0%" style="stop-color:{self.colors['accent']};stop-opacity:0.1" />
      <stop offset="100%" style="stop-color:{self.colors['accent']};stop-opacity:0.3" />
    </linearGradient>
  </defs>
  
  <path d="M0 50 Q300 0 600 50 T1200 50 L1200 150 L0 150 Z" fill="url(#waveGradient)">
    <animate attributeName="d" 
             values="M0 50 Q300 0 600 50 T1200 50 L1200 150 L0 150 Z;
                     M0 30 Q300 80 600 30 T1200 30 L1200 150 L0 150 Z;
                     M0 50 Q300 0 600 50 T1200 50 L1200 150 L0 150 Z"
             dur="10s" repeatCount="indefinite"/>
  </path>
  
  <path d="M0 80 Q300 30 600 80 T1200 80 L1200 150 L0 150 Z" fill="url(#waveGradient)" opacity="0.7">
    <animate attributeName="d" 
             values="M0 80 Q300 30 600 80 T1200 80 L1200 150 L0 150 Z;
                     M0 60 Q300 110 600 60 T1200 60 L1200 150 L0 150 Z;
                     M0 80 Q300 30 600 80 T1200 80 L1200 150 L0 150 Z"
             dur="8s" repeatCount="indefinite"/>
  </path>
</svg>'''
        
        self._save_svg('footer-wave.svg', wave_svg)
    
    def _save_svg(self, filename: str, content: str):
        """Save SVG content to file"""
        filepath = self.assets_path / filename
        filepath.parent.mkdir(parents=True, exist_ok=True)
        filepath.write_text(content)
        print(f"  ✅ {filename}")
    
    def create_readme_template(self):
        """Create a template README with correct asset paths"""
        readme_template = '''# TuskLang Python SDK - Asset Integration Guide

## 🎯 Asset Usage

All assets have been generated in the `assets/` directory. Update your README.md with these paths:

### Logo
```markdown
<img src="assets/tusklang-logo.svg" alt="TuskLang" width="600" />
```

### Hero Banner
```markdown
<img src="assets/hero-banner.svg" alt="TuskLang Hero" width="100%" />
```

### Function Dashboard
```markdown
<img src="assets/function-dashboard.svg" alt="Function Dashboard" width="90%" />
```

### Icons
```markdown
<img src="assets/icons/functions.svg" width="80" />
<img src="assets/icons/operators.svg" width="80" />
<img src="assets/icons/enterprise.svg" width="80" />
<img src="assets/icons/performance.svg" width="80" />
```

### Performance Chart
```markdown
<img src="assets/performance-comparison.svg" alt="Performance Comparison" width="80%" />
```

## 📁 Generated Files

'''
        
        # List all generated files
        for svg_file in sorted(self.assets_path.rglob("*.svg")):
            relative_path = svg_file.relative_to(self.base_path)
            readme_template += f"- `{relative_path}`\n"
        
        readme_path = self.base_path / "ASSET_INTEGRATION.md"
        readme_path.write_text(readme_template)
        print(f"\n📄 Created asset integration guide: {readme_path}")
    
    def generate_all(self):
        """Generate all assets"""
        print("\n🚀 TuskLang Asset Generator")
        print("=" * 50)
        
        # Setup directories
        self.setup_directories()
        
        # Generate main assets
        print("\n📄 Generating main assets:")
        self.create_logo()
        self.create_hero_banner()
        self.create_function_dashboard()
        self.create_performance_chart()
        self.create_quickstart_flow()
        self.create_footer_wave()
        
        # Generate icons
        print("\n🎨 Generating icons:")
        self.create_icons()
        
        # Create integration guide
        self.create_readme_template()
        
        print("\n" + "=" * 50)
        print("✨ Asset generation complete!")
        print(f"📁 Assets location: {self.assets_path.absolute()}")
        print("\n📝 Next steps:")
        print("1. Review generated assets in the assets/ directory")
        print("2. Read ASSET_INTEGRATION.md for usage instructions")
        print("3. Update your README.md with the new asset paths")
        print("4. Commit and push to your repository")
        print("\n💡 Tip: For GitHub hosting, use:")
        print("   https://raw.githubusercontent.com/[username]/[repo]/main/assets/[filename].svg")


def main():
    """Main entry point"""
    import argparse
    
    parser = argparse.ArgumentParser(
        description='Generate SVG assets for TuskLang Python SDK documentation',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog='''
Examples:
  python generate_tusklang_assets.py              # Generate in current directory
  python generate_tusklang_assets.py --path ../   # Generate in parent directory
  
This will create an 'assets/' directory with all SVG files organized by type.
        '''
    )
    
    parser.add_argument(
        '--path', 
        default='.', 
        help='Base path for asset generation (default: current directory)'
    )
    
    args = parser.parse_args()
    
    # Create generator and run
    generator = TuskLangAssetGenerator(args.path)
    generator.generate_all()


if __name__ == '__main__':
    main()
